<?php
/**
 * Plugin Name: Plugin Remover
 * Plugin URI: https://example.com/plugin-remover
 * Description: Deactivates and deletes ALL other plugins. Use with caution! This plugin will self-destruct after removing all others.
 * Version: 1.0.0
 * Author: Your Name
 * License: GPL v2 or later
 * Text Domain: plugin-remover
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Plugin_Remover {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function __construct() {
        add_action('admin_init', array($this, 'remove_all_plugins'));
        add_action('admin_notices', array($this, 'show_notice'));
    }
    
    public function remove_all_plugins() {
        // Only run once
        if (get_option('plugin_remover_completed')) {
            return;
        }
        
        // Get this plugin's basename
        $this_plugin = plugin_basename(__FILE__);
        
        // Get all active plugins
        $active_plugins = get_option('active_plugins', array());
        
        // Deactivate all plugins except this one
        $plugins_to_deactivate = array();
        foreach ($active_plugins as $plugin) {
            if ($plugin !== $this_plugin) {
                $plugins_to_deactivate[] = $plugin;
            }
        }
        
        // Deactivate them
        if (!empty($plugins_to_deactivate)) {
            deactivate_plugins($plugins_to_deactivate, true);
        }
        
        // Get all installed plugins
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $all_plugins = get_plugins();
        $deleted_plugins = array();
        $failed_plugins = array();
        
        // Delete all plugin folders except this one
        foreach ($all_plugins as $plugin_file => $plugin_data) {
            if ($plugin_file === $this_plugin) {
                continue;
            }
            
            // Get plugin folder
            $plugin_dir = WP_PLUGIN_DIR . '/' . dirname($plugin_file);
            $plugin_single_file = WP_PLUGIN_DIR . '/' . $plugin_file;
            
            // Check if it's a folder or single file plugin
            if (dirname($plugin_file) !== '.') {
                // It's in a folder
                if (is_dir($plugin_dir)) {
                    if ($this->delete_directory($plugin_dir)) {
                        $deleted_plugins[] = $plugin_data['Name'];
                    } else {
                        $failed_plugins[] = $plugin_data['Name'];
                    }
                }
            } else {
                // It's a single file plugin
                if (file_exists($plugin_single_file)) {
                    if (@unlink($plugin_single_file)) {
                        $deleted_plugins[] = $plugin_data['Name'];
                    } else {
                        $failed_plugins[] = $plugin_data['Name'];
                    }
                }
            }
        }
        
        // Store results
        update_option('plugin_remover_completed', true);
        update_option('plugin_remover_deleted', $deleted_plugins);
        update_option('plugin_remover_failed', $failed_plugins);
    }
    
    private function delete_directory($dir) {
        if (!is_dir($dir)) {
            return false;
        }
        
        $files = array_diff(scandir($dir), array('.', '..'));
        
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            if (is_dir($path)) {
                $this->delete_directory($path);
            } else {
                @unlink($path);
            }
        }
        
        return @rmdir($dir);
    }
    
    public function show_notice() {
        if (!get_option('plugin_remover_completed')) {
            return;
        }
        
        $deleted = get_option('plugin_remover_deleted', array());
        $failed = get_option('plugin_remover_failed', array());
        
        echo '<div class="notice notice-success is-dismissible">';
        echo '<p><strong>Plugin Remover:</strong> All plugins have been processed!</p>';
        
        if (!empty($deleted)) {
            echo '<p><strong>Deleted (' . count($deleted) . '):</strong> ' . esc_html(implode(', ', $deleted)) . '</p>';
        }
        
        if (!empty($failed)) {
            echo '<p><strong>Failed to delete:</strong> ' . esc_html(implode(', ', $failed)) . '</p>';
        }
        
        echo '<p><strong>Next steps:</strong></p>';
        echo '<ol>';
        echo '<li>Upload and install your new plugin (CF Security Shield)</li>';
        echo '<li>Activate your new plugin</li>';
        echo '<li>Delete this Plugin Remover plugin</li>';
        echo '</ol>';
        echo '</div>';
    }
}

// Initialize
Plugin_Remover::get_instance();
