<?php
/**
 * WordPress root cleanup cron script
 * - Force override index.php
 * - Delete wp-file-manager plugin if exists
 * - Clean unwanted root files
 */

// ----------------- CONFIG -----------------
$root = __DIR__;
$logFile = $root . '/cleanup.log';
$dryRun = true;

// Dangerous plugin path
$dangerPlugin = $root . '/wp-content/plugins/wp-file-manager';

// ----------------- WHITELIST -----------------
$whitelist = [
    'wp-admin',
    'wp-content',
    'wp-includes',
    '.htaccess',
    'error_log',
    'index.php',
    'license.txt',
    'readme.html',
    'wp-activate.php',
    'wp-blog-header.php',
    'wp-comments-post.php',
    'wp-config.php',
    'wp-config-sample.php',
    'wp-cron.php',
    'wp-links-opml.php',
    'wp-load.php',
    'wp-login.php',
    'wp-mail.php',
    'wp-settings.php',
    'wp-signup.php',
    'wp-trackback.php',
    'xmlrpc.php',
    basename(__FILE__),
    'cleanup.log'
];

// ----------------- CLI FLAGS -----------------
foreach ($argv ?? [] as $arg) {
    if ($arg === '--run') $dryRun = false;
}

// ----------------- HELPERS -----------------
function log_msg($msg)
{
    global $logFile;
    file_put_contents($logFile, date('[Y-m-d H:i:s] ') . $msg . PHP_EOL, FILE_APPEND);
}

function rrmdir_or_delete($path)
{
    if (!file_exists($path)) return;

    if (is_dir($path)) {
        foreach (scandir($path) as $file) {
            if ($file === '.' || $file === '..') continue;
            rrmdir_or_delete($path . '/' . $file);
        }
        rmdir($path);
    } else {
        unlink($path);
    }
}

// ----------------- STEP 1: FORCE OVERRIDE index.php -----------------
$indexContent = <<<PHP
<?php
define( 'WP_USE_THEMES', true );
require __DIR__ . '/wp-blog-header.php';
PHP;

if ($dryRun) {
    echo "[DRY-RUN] Would override index.php\n";
    log_msg("DRY-RUN Would override index.php");
} else {
    file_put_contents($root . '/index.php', $indexContent);
    echo "index.php overridden\n";
    log_msg("index.php overridden");
}

// ----------------- STEP 2: DELETE wp-file-manager PLUGIN -----------------
if (file_exists($dangerPlugin)) {
    if ($dryRun) {
        echo "[DRY-RUN] Would DELETE plugin: wp-file-manager\n";
        log_msg("DRY-RUN Would DELETE wp-file-manager");
    } else {
        rrmdir_or_delete($dangerPlugin);
        echo "Deleted plugin: wp-file-manager\n";
        log_msg("Deleted plugin: wp-file-manager");
    }
}

// ----------------- STEP 3: CLEAN ROOT -----------------
$items = scandir($root);
$flagged = 0;

foreach ($items as $item) {
    if ($item === '.' || $item === '..') continue;
    if (in_array($item, $whitelist)) continue;

    $full = $root . '/' . $item;
    $flagged++;

    if ($dryRun) {
        echo "[DRY-RUN] Would DELETE: $item\n";
        log_msg("DRY-RUN Would DELETE: $item");
    } else {
        rrmdir_or_delete($full);
        echo "DELETED: $item\n";
        log_msg("Deleted: $item");
    }
}

echo "\nScan complete. Items flagged: $flagged\n";
log_msg("Scan complete. Items flagged: $flagged");

if ($dryRun) {
    echo "Dry-run only. Use --run to perform deletion.\n";
}
